<?php

namespace App\Http\Controllers\Admin\Accounting;

use App\Http\Controllers\Controller;
use App\Http\Resources\Accounting\AssetResource;
use App\Models\Accounting\Asset;
use App\Models\Accounting\Depreciation;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class AssetController extends Controller
{
    public function index(Request $request)
    {
        $per_page = $request->per_page == null ? 25 : $request->per_page;
        $data = Asset::search($request->search_text)
            ->orderBy('id', 'DESC')
            ->paginate($per_page);
        return AssetResource::collection($data);
    }

    public function create(Request $request)
    {
        $asset = Asset::find($request->id);
        if ($asset) {
            $res = [
                'status'  => 1,
                'message' => 'Get Asset success',
                'data'    => new AssetResource($asset),
            ];
        } else {
            $res = [
                'status'  => 0,
                'message' => 'Get Asset not found',
                'data'    => [],
            ];
        }
        return response()->json($res, 200);
    }

    public function store(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|unique:assets,name',
        ]);
        try {
            DB::beginTransaction();
            $asset = Asset::updateOrCreate(['id' => $this->getRequest($request)['id']], $this->getRequest($request));
            DB::commit();
            $res = [
                'status' => 1,
                'message' => $asset->wasRecentlyCreated ? 'Create Asset success' : 'Update Asset success',
                'data'   => new AssetResource($asset),
            ];
        } catch (Exception $e) {
            DB::rollBack();
            $res = [
                'status'  => 0,
                'message' => 'Create Asset not success',
                'data'    => $e->getMessage(),
            ];
        }
        return response()->json($res, 200);
    }

    public function getRequest(Request $request)
    {
        $data = [
            'id'         => $request->id,
            'name'       => $request->name,
            'desc'       => $request->desc,
            'created_by' => Auth::user()->id,
        ];
        return $data;
    }
    
    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $asset = Asset::find($id);
            if ($asset) {
                $dep = Depreciation::where('asset_id', $id)->count();
                if ($dep > 0) {
                    $res = [
                        'status'  => 0,
                        'message' => 'Can not Delete!',
                        'data'    => $asset,
                    ];
                } else {
                    $asset->delete();
                    DB::commit();
                    $res = [
                        'status'  => 1,
                        'message' => 'Deleted success',
                        'data'    => $asset,
                    ];
                }
            }
        } catch (Exception $e) {
            DB::rollBack();
            $res = [
                'status'  => 0,
                'message' => 'Deleted not success',
                'data'    => $e->getMessage(),
            ];
        }
        return response()->json($res, 200);
    }
}
