<?php

namespace App\Http\Controllers\Admin\Accounting;

use App\Http\Controllers\Controller;
use App\Http\Resources\Accounting\ChartOfAccountResource;
use Illuminate\Validation\Rule;
use App\Models\Accounting\ChartOfAccount;
use App\Models\Accounting\Journal;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ChartOfAccountController extends Controller
{
    public function index(Request $request)
    {
        $per_page = $request->per_page == null ? 25 : $request->per_page;
        $data = ChartOfAccount::search($request->search_text)
            ->orderby('chart_code', 'ASC')
            // ->orderby('id', 'ASC')
            ->paginate($per_page);
        return ChartOfAccountResource::collection($data);
    }

    public function create(Request $request)
    {
        $chartAcc = ChartOfAccount::find($request->id);
        if ($chartAcc) {
            $res = [
                'status'  => 1,
                'message' => 'Get ChartOfAccount success',
                'data'    => new ChartOfAccountResource($chartAcc),
            ];
        } else {
            $res = [
                'status'  => 0,
                'message' => 'Get ChartOfAccount not found',
                'data'    => [],
            ];
        }
        return response()->json($res, 200);
    }

    public function store(Request $request)
    {
        $this->validate($request, [
            'type_of_acc_id' => 'required',
            'chart_name'     => 'required|max:150',
            'type_of_acc_detail_id' => 'required',
            'chart_code' => [
                'required',
                'numeric',
                'between:10000,59999',
                Rule::unique('chart_of_accounts')->where(function ($query) {
                    $query->where('deleted_at', '=', NUlL);
                })->ignore(request()->id),
            ],
        ]);
        DB::beginTransaction();
        try {
            $chartAcc = ChartOfAccount::updateOrCreate(
                ['id' => $request->id],
                [
                    'chart_name' => $request->chart_name,
                    'chart_code' => $request->chart_code,
                    'desc'       => $request->desc,
                    'created_by' => Auth::user()->id,
                    'type_of_acc_detail_id' => $request->type_of_acc_detail_id,
                ]
            );
            DB::commit();
            $res = [
                'status'  => 1,
                'message' => $chartAcc->wasRecentlyCreated ? 'Create ChartOfAccount success' : 'Update ChartOfAccount success',
                'data'    => new ChartOfAccountResource($chartAcc),
            ];
        } catch (\Exception $e) {
            DB::rollback();
            $res = [
                'status'  => 0,
                'message' => 'Create ChartOfAccount not success',
                'data'    => $e->getMessage(),
            ];
        }
        return response()->json($res, 200);
    }

    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $chartAcc = ChartOfAccount::find($id);
            if ($chartAcc) {
                $journal = Journal::where('chart_of_acc_id', $id)->count();
                if ($journal > 0) {
                    $res = [
                        'status'  => 0,
                        'message' => 'Can not Delete!',
                        'data'    => $chartAcc,
                    ];
                } else {
                    $chartAcc->delete();
                    DB::commit();
                    $res = [
                        'status'  => 1,
                        'message' => 'Deleted success',
                        'data'    => $chartAcc,
                    ];
                }
            }
        } catch (Exception $e) {
            DB::rollBack();
            $res = [
                'status'  => 0,
                'message' => 'Deleted not success',
                'data'    => $e->getMessage(),
            ];
        }
        return response()->json($res, 200);
    }
}
