<?php

namespace App\Http\Controllers\Admin\Accounting;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Resources\Accounting\DepreciationResource;
use App\Models\Accounting\Depreciation;
use App\Models\Accounting\DepreciationDetail;
use App\Models\Accounting\DepreciationPayment;
use App\Models\Accounting\Journal;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DepreciationController extends Controller
{
   
    public function index(Request $request)
    {
        $per_page = $request->per_page == null ? 25 : $request->per_page;
        $data = Depreciation::search($request->search_text)
            ->whereIn('branch_id',Helper::byBranch())
            ->where('currency_id',Helper::byCurrency())
            ->orderBy('id', 'DESC')
            ->paginate($per_page);
        return DepreciationResource::collection($data);
    }

    public function store(Request $request)
    {
        $this->validate($request, [
            'asset_id'      => 'required',
            'amount'        => 'required',
            'chart_asset'   => 'required',
            'chart_expense' => 'required',
            'payment_date'  => 'required',
            'depreciation_date' => 'required',
        ]);
        try {
            DB::beginTransaction();
            $count_period = count($request->arr_depreciation);
            $depreciation = Depreciation::create([
                'asset_id'          => $request->asset_id,
                'total_amount'      => $request->amount,
                'total_paid'        => 0,
                'balance'           => $request->amount,
                'period'            => $count_period,
                'chart_asset'       => $request->chart_asset,
                'chart_expense'     => $request->chart_expense,
                'desc'              => $request->desc,
                'branch_id'        => Auth::user()->branch_id,
                'currency_id'        => Auth::user()->currency_id,
                'created_by'        => Auth::user()->id,
                'depreciation_date' => $request->depreciation_date,
            ]);
            foreach($request->arr_depreciation as $key => $value){
                $arr = array(
                    'dep_id'          => $depreciation->id,
                    'payment_date'    => $value['payment_date'],
                    'amount'          => $value['payment_amount'],
                    'paid'            => 0,
                    'unpaid'          => $value['payment_amount'],
                    'loan_balance'    => $value['loan_balance'],
                    'status'          => 1,
                    'branch_id'        => Auth::user()->branch_id,
                    'currency_id'        => Auth::user()->currency_id,
                );
                DepreciationDetail::insert($arr);
            }
            // $parent_id = Helper::findParentId();
            // $journal   = $this->insertJournal($depreciation->chart_asset, $parent_id, date('Y-m-d'), $depreciation->id, get_class($depreciation),$request->amount,0);
            // $parentID  = $journal->parent_id;
            // $this->insertJournal($depreciation->chart_expense, $parentID, date('Y-m-d'), $depreciation->id, get_class($depreciation),0,$request->amount);
            DB::commit();
            $res = [
                'status'  => 1,
                'message' => 'Create Depreciation Successfully',
            ];
        }catch (Exception $e) {
            DB::rollBack();
            $res = [
                'status'  => 0,
                'message' => 'Create Depreciation not success',
                'data'    => $e->getMessage(),
            ];
        }
        return response()->json($res, 200);
    }

    public function show($id)
    {
        $data = Depreciation::find($id);
        return new DepreciationResource($data);
    }

    public function insertJournal($chartID,$parentID,$journalDate,$paymentTable,$paymentTableType,$debit,$credit)
    {   return Journal::create([
            'chart_of_acc_id'   => $chartID,
            'parent_id'         => $parentID,
            'journal_date'      => $journalDate,
            'payment_model_id'  => $paymentTable,
            'payment_model_type'=> $paymentTableType,
            'debit'             => $debit,
            'credit'            => $credit,
            'branch_id'        => Auth::user()->branch_id,
            'currency_id'        => Auth::user()->currency_id,
            'created_by'        => Auth::user()->id,
        ]);
    }

    public function submitPaymentDepreciation($id)
    {
        DB::beginTransaction();
        try {
            $depDetails = DepreciationDetail::find($id);
            $depDetails->status = 2;
            $depDetails->paid_date = date('Y-m-d');
            $depDetails->increment('paid',$depDetails->unpaid);
            $depDetails->decrement('unpaid',$depDetails->unpaid);
            $depDetails->update();

            $depreciation = Depreciation::find($depDetails->dep_id);
            $depreciation->increment('total_paid',$depDetails->paid);
            $depreciation->decrement('balance',$depDetails->paid);
            $depreciation->update();

            $depPayment = DepreciationPayment::create([
                'dep_detail_id' => $depDetails->id,
                'paid_date'     => date('Y-m-d'),
                'paid'          => $depDetails->paid,
                'loan_balance'  => $depreciation->balance,
                'created_by'    => Auth::user()->id,
                'branch_id'        => Auth::user()->branch_id,
                'currency_id'        => Auth::user()->currency_id,
            ]);
            $parent_id = Helper::findParentId();
            $journal   = $this->insertJournal($depreciation->chart_asset, $parent_id, date('Y-m-d'), $depPayment->id, get_class($depPayment),0,$depDetails->paid);
            $parentID  = $journal->parent_id;
            $this->insertJournal($depreciation->chart_expense, $parentID, date('Y-m-d'), $depPayment->id, get_class($depPayment),$depDetails->paid,0);
            DB::commit();
        } 
        catch (\Exception $e) {
            return response()->json(["message" => "Something wrong! please check it again"], 200);
        }
        return response()->json(["message" => "Depreciation's Pay Successfully."], 200);
    }

    public function getMonthlyDepreciation(Request $request)
    {
        $arr_assset_id = $request->asset_id;
        $per_page = $request->per_page == null ? 25 : $request->per_page;
        $getDepre = DepreciationDetail::where('status',1)
        ->whereMonth('payment_date',date('m'))
        ->whereYear('payment_date',date('Y'))
        ->whereIn('branch_id',Helper::byBranch())
        ->where('currency_id',Helper::byCurrency())
        ->whereHas('depreciation', function ($query) use ($arr_assset_id) {
            $query->whereHas('assets',function($q) use($arr_assset_id){
                if(!$arr_assset_id == []){
                    $q->whereIn('id',$arr_assset_id);
                }
            });
        })
        ->orderby('payment_date','desc')
        ->paginate($per_page);

        $page =  json_encode($getDepre);
        $paginate = json_decode($page);

        $data = $getDepre->map(function($value){
            return [
                'id'     => $value->id,
                'amount' => $value->amount,
                'payment_date'    => $value->payment_date,
                'asset_name'      => $value->depreciation->assets->name??'',
            ];
        });
        $res = [
            'data'  => $data,
            'link'  => [
                'first'=> $paginate->first_page_url,
                'last' => $paginate->last_page_url,
                'next' => $paginate->next_page_url,
                'prev' => $paginate->prev_page_url,
            ],
            'meta'  => [
                'current_page'  => $getDepre->currentPage(),
                'last_page'     => $getDepre->lastPage(),
                'per_page'      => $getDepre->perPage(),
                'total'         => $getDepre->total(),
                'to'            => $paginate->to,
            ],
        ];
        return response()->json($res, 200);
    }
    
    public function submitAllPaymentDepreciation(Request $request)
    {
        DB::beginTransaction();
        try {
            $depDetailsArrayID = $request->arrayID;
            $arrayDepDetails = DepreciationDetail::whereIn('id',$depDetailsArrayID)->get();
            foreach($arrayDepDetails as $key => $value){
                $depPayment = DepreciationPayment::create([
                    'dep_detail_id' => $value['id'],
                    'paid_date'     => date('Y-m-d'),
                    'paid'          => $value['unpaid'],
                    'loan_balance'  => $value['loan_balance'],
                    'created_by'    => Auth::user()->id,
                    'branch_id'        => Auth::user()->branch_id,
                    'currency_id'        => Auth::user()->currency_id,
                ]);
                DepreciationDetail::where('id',$value['id'])->update([
                    'paid_date' => date('Y-m-d'),
                    'status'    => 2,
                ]);

                Depreciation::where('id',$value['dep_id'])->increment('total_paid',$value['unpaid']);
                Depreciation::where('id',$value['dep_id'])->decrement('balance',$value['unpaid']);

                $deps = Depreciation::where('id',$value['dep_id'])->select('chart_asset','chart_expense')->first();

                $parent_id = Helper::findParentId();
                $journal   = $this->insertJournal($deps->chart_asset, $parent_id, date('Y-m-d'), $depPayment->id, get_class($depPayment),0,$value['unpaid']);
                $parentID  = $journal->parent_id;
                $this->insertJournal( $deps->chart_expense, $parentID, date('Y-m-d'), $depPayment->id, get_class($depPayment),$value['unpaid'],0);
                
                DepreciationDetail::where('id',$value['id'])->increment('paid',$value['unpaid']);
                DepreciationDetail::where('id',$value['id'])->decrement('unpaid',$value['unpaid']);
            }
            DB::commit();
        } 
        catch (\Exception $e) {
            return response()->json(["message" => "Something wrong! please check it again"], 200);
        }
        return response()->json(["message" => "Depreciation's Pay All Successfully."], 200);
    }
}
