<?php

namespace App\Http\Controllers\Admin\Accounting;

use App\Helpers\CustomDateFilter;
use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Models\Accounting\Journal;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class JournalController extends Controller
{
    public function getJournal(Request $request){
        $data = Journal::where(function($query) use ($request){
            $query->whereIn('branch_id',Helper::byBranch());
            $query->where('currency_id',Helper::byCurrency());
            if($request->journal_list == 1){
                $query->search($request->search_text);
            }
            else{
                $chart_of_acc_id = $request->chart_of_acc_id;
                if(!$chart_of_acc_id ==[]){
                    $query->whereIn('journals.chart_of_acc_id',$chart_of_acc_id);
                }
                $query->where('journals.invoice_number', 'LIKE',isset($request->invoice_number)? $request->invoice_number:NULL);
                if(isset($request->journal_date)){
                    $query->whereBetween(DB::raw("DATE_FORMAT(journal_date, '%Y-%m-%d')"), $request->journal_date ?? CustomDateFilter::thisMonth());
                }
            }
        })
        ->orderBy('journals.id','DESC')
        ->orderBy('journals.journal_date','DESC')
        ->groupby('journals.parent_id');
        return $data;
    }

    public function index(Request $request)
    {
        $chart_of_acc_id = $request->chart_of_acc_id;
        $per_page = $request->per_page == null?25:$request->per_page;
        $funGetJournal = $this->getJournal($request)->paginate($per_page);

        $page =  json_encode($funGetJournal);
        $paginate = json_decode($page);
        $data = $funGetJournal->map(function($value) use($chart_of_acc_id){
            return [
                'id' => $value->id,
                'parent_id'     => $value->parent_id,
                'invoice_number'=> $value->invoice_number,
                'journal_date'  => $value->journal_date,
                'desc'          => $value->desc,
                'debit'         => $value->debit,
                'credit'        => $value->credit,
                'payment_model_id' => $value->payment_model_id,
                'chart_code'    => $value->chartOfAcc->chart_code,
                'chart_name'    => $value->chartOfAcc->chart_name,
                'created_by'    => $value->createdBy->name,
                'sub_journal'   => Journal::where('journals.parent_id',$value->parent_id)
                ->where(function($query) use($chart_of_acc_id){
                    // if(!$chart_of_acc_id ==[]){
                    //     $query->whereIn('journals.chart_of_acc_id',$chart_of_acc_id);
                    // }
                })
                ->get()
                ->map(function($val){
                    return [
                        'id'         => $val->id,
                        'debit'      => $val->debit,
                        'credit'     => $val->credit,
                        'parent_id'  => $val->parent_id,
                        'chart_code' => $val->chartOfAcc->chart_code,
                        'chart_name' => $val->chartOfAcc->chart_name,
                    ];
                }),
                'sum_debit'   => Journal::where('journals.parent_id',$value->parent_id)->sum('debit'),
                'sum_credit'   => Journal::where('journals.parent_id',$value->parent_id)->sum('credit'),
            ];
        });

        $res = [
            'data'  => $data,
            'links'    => [
                'first'=> $paginate->first_page_url,
                'last' => $paginate->last_page_url,
                'next' => $paginate->next_page_url,
                'prev' => $paginate->prev_page_url,
            ],
            'meta'   => [
                'current_page'  => $funGetJournal->currentPage(),
                'last_page'     => $funGetJournal->lastPage(),
                'per_page'      => $funGetJournal->perPage(),
                'total'         => $funGetJournal->total(),
                'to'            => $paginate->to,
            ],
        ];
        return response()->json($res, 200);
    }

    public function store(Request $request)
    {
        $request->validate([
            'desc'                 => 'required',
            'invoice_number'       => 'required|max:255',
            'journal_date'         => 'required',
            'journal_rows.*.credit'=> 'required',
            'journal_rows.*.debit' => 'required',
            'journal_rows.*.chart_of_acc_id' => 'required',
        ]);
        DB::beginTransaction();
        try {
            $journal = $request->journal_rows;
            $parent_id = Helper::findParentId();
            
            foreach($journal as $value){
                $arr=array(
                    'parent_id'       => $parent_id,
                    'branch_id'       => Auth::user()->branch_id,
                    'currency_id'       => Auth::user()->currency_id,
                    'debit'           => $value['debit'],
                    'credit'          => $value['credit'],
                    'desc'            => $request->desc,
                    'created_by'      => Auth::user()->id,
                    'invoice_number'  => $request->invoice_number,
                    'chart_of_acc_id' => $value['chart_of_acc_id'],
                    'journal_date'    => $request->journal_date,
                );
                Journal::insert($arr);
            }
            DB::commit();
            return response()->json(["message" => "$request->invoice_number successfully created."], 200);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(["message" => "Something wrong, Please check it !"], 200);
        }
    }

    public function show($id)
    {
        $data = Journal::select(
            'journals.*',
            'chart_of_accounts.chart_name',
            'chart_of_accounts.chart_code',
        )
        ->leftjoin('chart_of_accounts','chart_of_accounts.id','journals.chart_of_acc_id')
        ->where('journals.parent_id',$id)
        ->orderBy('journals.id','ASC')
        ->get();
        return $data;
    }
    public function update(Request $request, $id)
    {
        $request->validate([
            'desc'          =>'required',
            'invoice_number'=>'required|max:255',
            'journal_date'  =>'required',
            'journal_rows.*.credit'=> 'required',
            'journal_rows.*.debit' => 'required',
            'journal_rows.*.chart_of_acc_id' => 'required',

        ]);
        DB::beginTransaction();
        try {
            $journalID = [];
            foreach($request->journal_rows as $value){
                $journals = Journal::updateOrCreate(
                    ['id' => $value['id']]
                    ,
                    [
                        'parent_id'       => $value['parent_id'],
                        'debit'           => $value['debit'],
                        'credit'          => $value['credit'],
                        'desc'            => $request->desc,
                        'created_by'      => Auth::user()->id,
                        'invoice_number'  => $request->invoice_number,
                        'chart_of_acc_id' => $value['chart_of_acc_id'],
                        'journal_date'    => date('Y-m-d',strtotime($request->journal_date)),
                    ]
                );
                $journalID [] = $journals->id;
            }
            DB::commit();
            return response()->json(["message" => "$request->invoice_number successfully created."], 200);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(["message" => "Something wrong, Please check it !"], 200);
        }
    }
    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            Journal::where('parent_id',$id)->delete();
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(["message" => "Something wrong, Please check it !"], 200);
        }
        return response()->json(["message" => "Journal deleted Successfully."], 200);
    }
}
